﻿using System;
using System.Configuration;
using System.IO;
using System.Linq;
using System.Xml.Linq;
using gov.va.med.vbecs.Common.AppServices;

namespace gov.va.med.vbecs.Common.Configuration
{
    /// <summary>
    /// DbSettingsReaderXml reads database information from XML file
    /// </summary>
    public class DbSettingsReaderXml : IDbSettingsReader
    {
        //CR2113 - removed Connection Timeout otherwise the commandTimeout value set in StoredProcedure is ignored
        private const string DB_CONN_STRING_TEMPLATE = "Data Source=XXXSQLSERVERXXX;;;Initial Catalog=XXXDATABASEXXX;persist security info=False;packet size=8192;Application Name=XXXAPPLICATIONXXX;Integrated Security=SSPI;";

        private string _primaryDbConnectionString;

        /// <summary>
        /// PrimaryDbConnectionString
        /// </summary>
        public string PrimaryDbConnectionString
        {
            get
            {
                if (_primaryDbConnectionString == null)
                    read_db_settings();
                return _primaryDbConnectionString;
            }
        }

        /// <summary>
        /// DataSourceString: file name to read values. Assigned by spring .NET.
        /// </summary>
        public string DataSourceString { get; internal set; }

        /// <summary>
        /// AppSettingsReaderXml
        /// </summary>
        internal DbSettingsReaderXml()
        {}

        private void read_db_settings()
        {
            if (DataSourceString == null)
                throw new ArgumentNullException(DataSourceString);

            try
            {
                // Check if path rooted.
                XDocument xdoc;
                string envDesc = "-NonStandard";
                if (!Path.IsPathRooted(DataSourceString))
                {
                    // If path is relative, try to find document relative from current folder
#if NUNIT
                    // TODO: remove NUNIT directive and use rooted paths for unit tests.
                    xdoc = XDocument.Load(Path.Combine(Environment.CurrentDirectory ?? ".", DataSourceString));
                    envDesc = "-NUnit";
#else
                    string assemblyPath = typeof(DbSettingsReaderXml).Assembly.Location;

                    xdoc =
                        XDocument.Load(
                            Path.Combine(Path.GetDirectoryName(assemblyPath) ?? ".", DataSourceString));

                    //CR2113 -adding more detail to the connection string for sp_who2 troubleshooting
                    if (assemblyPath.ToUpper().Contains("TEST")) envDesc = "-Test";
                    if (assemblyPath.ToUpper().Contains("PROD")) envDesc = "-Prod";
#endif
                }
                else
                {
                    // If path rooted - load XML from there
                    xdoc = XDocument.Load(DataSourceString);
                }

                var databaseElement = xdoc.Descendants("database").SingleOrDefault();
                if (databaseElement == null)
                    throw new ConfigurationErrorsException("database configuration is not found");

// ReSharper disable PossibleNullReferenceException
                var serverName = databaseElement.Descendants("serverName").SingleOrDefault().Value;
                var dataBaseName = databaseElement.Descendants("databaseName").SingleOrDefault().Value + getDataBaseSuffix();
                
                //CR3522 - adding connectionOptions - which can be optional and null check for backward compatibility
                var connOptElement = databaseElement.Descendants("connectionOptions").SingleOrDefault();
                var connectionOptions = (connOptElement == null) ? null : connOptElement.Value;

                //CR2113 -adding more detail to the connection string for sp_who2 troubleshooting
                _primaryDbConnectionString = DB_CONN_STRING_TEMPLATE.Replace("XXXSQLSERVERXXX", serverName).Replace("XXXDATABASEXXX", dataBaseName).Replace("XXXAPPLICATIONXXX",AppDomain.CurrentDomain.FriendlyName.Replace(".exe",string.Empty) + envDesc);
// ReSharper restore PossibleNullReferenceException

                if (!string.IsNullOrEmpty(connectionOptions)) _primaryDbConnectionString += connectionOptions; //CR3522
                
            }
            catch (Exception ex)
            {
                throw new ConfigurationErrorsException("Database configuration is not found", ex);
            }
        }

        /// <summary>
        /// Derive this method to handle suffixes for different environments
        /// </summary>
        protected virtual string getDataBaseSuffix()
        {
            return "";
        }
    }
}
